package tech.starwin.service.sms;

import android.app.PendingIntent;
import android.content.BroadcastReceiver;
import android.content.ContentValues;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.provider.Telephony;
import android.support.v4.app.NotificationCompat;
import android.telephony.SmsMessage;

import tech.starwin.LibConfig;
import tech.starwin.R;
import tech.starwin.utils.LogUtils;
import tech.starwin.utils.ui_utils.UIHelper;

/**
 * Created by SiKang on 2018/11/5.
 * 短信接收器
 * 当用户同意将APP设置为默认短信应用后，手机收到短信第一时间会通知到这里
 * 由于系统短信应用已被取消默认权限，所有不会收到通知，保存短信的逻辑也不会执行，所以这里需要主动保存短信内容，否则用户打开短信列表将无法看见短信
 */
public class SmsReceiver extends BroadcastReceiver {
    @Override
    public void onReceive(Context context, Intent intent) {
        LibConfig.getContext().getPackageManager().clearPackagePreferredActivities(LibConfig.APPLICATION_ID);

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
            //如果是默认短信应用，收到短信主动保存到数据库
            if ("android.provider.Telephony.SMS_RECEIVED".equals(intent.getAction()) && Telephony.Sms.getDefaultSmsPackage(context).equals(context.getPackageName())) {
                Bundle bundle = intent.getExtras();
                if (bundle != null) {
                    Object[] pdus = (Object[]) bundle.get("pdus");
                    for (int i = 0; i < pdus.length; i++) {
                        //提取短信内容
                        SmsMessage msg = SmsMessage.createFromPdu((byte[]) pdus[i]);
                        String msgBody = msg.getMessageBody();
                        String msgAddress = msg.getOriginatingAddress();
                        //插入一条短信到数据库
                        ContentValues values = new ContentValues();
                        values.put("date", msg.getTimestampMillis());//发送时间
                        values.put("read", 0);//阅读状态 0：未读  1：已读
                        values.put("type", 1);//1为收 2为发
                        values.put("address", msgAddress);//送达号码
                        values.put("body", msgBody);//送达内容
                        context.getContentResolver().insert(Uri.parse("content://sms/sent"), values);//插入短信库
                        long msgDate = msg.getTimestampMillis();
                        LogUtils.d("cky", "message from: " + msgAddress + ", message body: " + msgBody + ", message date: " + msgDate);
                        //通知栏提醒
                        notifySms(context, msgAddress, msgBody);
                    }
                }
            }
        }
    }


    /**
     * 添加一条短信通知
     */
    public void notifySms(Context context, String title, String body) {
        //点击通知栏进入系统短信列表
        Intent intent = new Intent(Intent.ACTION_MAIN);
        intent.addCategory(Intent.CATEGORY_DEFAULT);
        intent.setType("vnd.android-dir/mms-sms");
        PendingIntent pendingIntent = PendingIntent.getActivity(context, 0, intent, 0);

        NotificationCompat.Builder builder = new NotificationCompat.Builder(context, LibConfig.APPLICATION_ID)
                .setContentTitle(title)
                // 设置通知时间，此事件用于通知栏排序
                .setContentText(body)
                .setSmallIcon(R.mipmap.ic_launcher)
                .setTicker(title)
                .setContentIntent(pendingIntent);
        UIHelper.notifyNotification(context, builder);

    }
}
