package tech.starwin.widget;


import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.AttributeSet;
import android.view.View;
import android.view.ViewGroup;
import android.widget.Button;
import android.widget.EditText;
import android.widget.FrameLayout;
import android.widget.ImageButton;
import android.widget.ImageView;
import android.widget.RelativeLayout;
import android.widget.TextView;

import java.util.HashMap;
import java.util.Map;


/**
 * Created by SiKang on 2018/10/12.
 * 页面状态展示  loading、错误、无数据、重新请求 等
 * 通过 setStateView（） 指定一个View，在view和父布局之间，嵌套一层FrameLayout,并加入自定义的展示界面
 */
public class PageStateLayout extends RelativeLayout {
    private View mStateView;
    private OnStateChangedListener onStateChangedListener;
    private OnRetryListener onRetryListener;
    private ViewHolder viewHolder;

    private PageStateLayout(@NonNull Context context) {
        super(context);
    }

    private PageStateLayout(@NonNull Context context, @Nullable AttributeSet attrs) {
        super(context, attrs);
    }

    private PageStateLayout(@NonNull Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
    }

    private void setStateView(@NonNull View view) {
        mStateView = view;
        viewHolder = new ViewHolder(view);
        addView(mStateView);
    }

    public void setOnStateChangedListener(OnStateChangedListener onStateChangedListener) {
        this.onStateChangedListener = onStateChangedListener;
    }

    public void setOnRetryListener(OnRetryListener onRetryListener) {
        this.onRetryListener = onRetryListener;
    }


    public void showError() {
        showStateView();
        if (onStateChangedListener != null) {
            onStateChangedListener.onError(viewHolder);
        }
    }

    public void showLoading() {
        showStateView();
        if (onStateChangedListener != null) {
            onStateChangedListener.onLoading(viewHolder);
        }
    }

    public void showEmpty() {
        showStateView();
        if (onStateChangedListener != null) {
            onStateChangedListener.onEmpty(viewHolder);
        }
    }

    public void retry() {
        hide();
        if (onRetryListener != null) {
            onRetryListener.onRetry();
        }
    }

    public void hide() {
        if (mStateView != null) {
            mStateView.setVisibility(GONE);
        }
    }

    private void showStateView() {
        if (mStateView != null) {
            mStateView.setVisibility(VISIBLE);
        }
    }

    public static class Builder {
        private PageStateLayout pageStateLayout;

        public Builder(View view) {
            //得到目标父布局，将view记录索引后，从布局中移除
            ViewGroup parent = (ViewGroup) view.getParent();
            int index = parent.indexOfChild(view);
            parent.removeView(view);

            //创建PageStateLayout,并添加到之前view所在的位置
            pageStateLayout = new PageStateLayout(view.getContext());
            parent.addView(pageStateLayout, index, view.getLayoutParams());
            pageStateLayout.addView(view);
        }

        public Builder setStateView(@NonNull View view) {
            pageStateLayout.setStateView(view);
            return this;
        }

        public Builder setOnStateChangedListener(OnStateChangedListener listener) {
            pageStateLayout.setOnStateChangedListener(listener);
            return this;
        }

        public Builder setOnRetryListener(OnRetryListener listener) {
            pageStateLayout.setOnRetryListener(listener);
            return this;
        }

        public PageStateLayout create() {
            return pageStateLayout;
        }

    }

    public static class ViewHolder {
        private View view;
        Map<Integer, View> childViews;

        public ViewHolder(View view) {
            childViews = new HashMap<>();
            this.view = view;
        }

        public View getContentView() {
            return view;
        }

        public View getView(int id) {
            if (childViews.containsKey(id)) {
                return childViews.get(id);
            } else {
                return view.findViewById(id);
            }
        }

        public TextView getTextView(int viewId) {
            return (TextView) getView(viewId);
        }

        public Button getButton(int viewId) {
            return (Button) getView(viewId);
        }

        public ImageView getImageView(int viewId) {
            return (ImageView) getView(viewId);
        }

        public ImageButton getImageButton(int viewId) {
            return (ImageButton) getView(viewId);
        }

        public EditText getEditText(int viewId) {
            return (EditText) getView(viewId);
        }

    }


    public interface OnStateChangedListener {
        void onLoading(ViewHolder holder);

        void onError(ViewHolder holder);

        void onEmpty(ViewHolder holder);
    }

    public interface OnRetryListener {
        void onRetry();

    }


}
