package tech.starwin.utils.context_utils;

import android.Manifest;
import android.annotation.SuppressLint;
import android.app.AppOpsManager;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.pm.ApplicationInfo;
import android.net.Uri;
import android.os.Build;
import android.provider.Settings;
import android.support.v4.app.Fragment;
import android.support.v4.app.FragmentActivity;

import com.tbruyelle.rxpermissions2.Permission;
import com.tbruyelle.rxpermissions2.RxPermissions;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;

import io.reactivex.functions.Consumer;
import tech.starwin.BuildConfig;

/**
 * Created by SiKang on 2018/9/20.
 * 权限管理
 */
public class PermissionsHelper {

    public static void checkCameraPermission(FragmentActivity activity, OnPermissionListener listener) {
        checkPermission(activity, new String[]{Manifest.permission.CAMERA,
                Manifest.permission.READ_EXTERNAL_STORAGE,
                Manifest.permission.WRITE_EXTERNAL_STORAGE,
                Manifest.permission.RECORD_AUDIO}, listener);
    }

    /**
     * 检查权限
     */
    public static void checkPermission(FragmentActivity activity, String[] permissions, final OnPermissionListener listener) {
        getPermission(new RxPermissions(activity), permissions, listener);
    }

    public static void checkPermission(Fragment fragment, String[] permissions, final OnPermissionListener listener) {
        getPermission(new RxPermissions(fragment), permissions, listener);
    }

    @SuppressLint("CheckResult")
    private static void getPermission(RxPermissions rxPermissions, final String[] permissions, final OnPermissionListener listener) {
        rxPermissions
                .requestEach(permissions)
                .subscribe(new Consumer<Permission>() {
                    List<Permission> allowed = new ArrayList<>();
                    List<Permission> refused = new ArrayList<>();
                    List<Permission> neverAsk = new ArrayList<>();

                    @Override
                    public void accept(Permission permission) throws Exception {
                        if (permission.granted) {
                            allowed.add(permission);
                            listener.onAllow(permission);
                        } else if (permission.shouldShowRequestPermissionRationale) {
                            refused.add(permission);
                            listener.onRefuse(permission);
                        } else {
                            neverAsk.add(permission);
                            listener.onRefuseAndNeverAskAgain(permission);
                        }

                        if (allowed.size() + refused.size() + neverAsk.size() == permissions.length) {
                            listener.onFinish(allowed.toArray(new Permission[allowed.size()]),
                                    refused.toArray(new Permission[refused.size()]),
                                    neverAsk.toArray(new Permission[neverAsk.size()]));
                        }


                    }
                });
    }

    /**
     * 去系统设置中设置权限
     */
    public static void goSystemSetting(Context context) {
        Intent localIntent = new Intent();
        localIntent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        if (Build.VERSION.SDK_INT >= 9) {
            localIntent.setAction("android.settings.APPLICATION_DETAILS_SETTINGS");
            localIntent.setData(Uri.fromParts("package", context.getPackageName(), null));
        } else if (Build.VERSION.SDK_INT <= 8) {
            localIntent.setAction(Intent.ACTION_VIEW);

            localIntent.setClassName("com.android.settings",
                    "com.android.settings.InstalledAppDetails");

            localIntent.putExtra("com.android.settings.ApplicationPkgName",
                    context.getPackageName());
        }
        context.startActivity(localIntent);
    }

    /**
     * 是否开启了通知权限
     */
    @SuppressLint("NewApi")
    public static boolean isNotificationEnabled(Context context) {

        AppOpsManager mAppOps =
                (AppOpsManager) context.getSystemService(Context.APP_OPS_SERVICE);

        ApplicationInfo appInfo = context.getApplicationInfo();
        String pkg = context.getApplicationContext().getPackageName();
        int uid = appInfo.uid;
        Class appOpsClass = null;

        try {
            appOpsClass = Class.forName(AppOpsManager.class.getName());

            Method checkOpNoThrowMethod =
                    appOpsClass.getMethod("checkOpNoThrow",
                            Integer.TYPE, Integer.TYPE, String.class);

            Field opPostNotificationValue = appOpsClass.getDeclaredField("OP_POST_NOTIFICATION");
            int value = (Integer) opPostNotificationValue.get(Integer.class);

            return ((Integer) checkOpNoThrowMethod.invoke(mAppOps, value, uid, pkg) ==
                    AppOpsManager.MODE_ALLOWED);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return false;
    }


    public static void gotoPermissionSetting(Context context) {
        try {
            Intent intent = null;
            switch (Build.MANUFACTURER.toUpperCase()) {
                case "HUAWEI":
                    intent = new Intent();
                    intent.setComponent(new ComponentName("com.huawei.systemmanager", "com.huawei.permissionmanager.ui.MainActivity"));
                    break;
                case "XIAOMI":
                    intent = new Intent("miui.intent.action.APP_PERM_EDITOR");
                    intent.setComponent(new ComponentName("com.miui.securitycenter", "com.miui.permcenter.permissions.AppPermissionsEditorActivity"));
                    break;
                case "SONY":
                    intent = new Intent();
                    intent.setComponent(new ComponentName("com.sonymobile.cta", "com.sonymobile.cta.SomcCTAMainActivity"));
                    break;
                case "OPPO":
                    intent = new Intent();
                    intent.setComponent(new ComponentName("com.color.safecenter", "com.color.safecenter.permission.PermissionManagerActivity"));
                    break;
                case "LG":
                    intent = new Intent("android.intent.action.MAIN");
                    intent.setComponent(new ComponentName("com.android.settings", "com.android.settings.Settings$AccessLockSummaryActivity"));
                    break;

                case "LETV":
                    intent = new Intent();
                    intent.setComponent(new ComponentName("com.letv.android.letvsafe", "com.letv.android.letvsafe.PermissionAndApps"));
                    break;
                case "360":
                case "QIKU":
                    intent = new Intent("android.intent.action.MAIN");
                    ComponentName comp = new ComponentName("com.qihoo360.mobilesafe", "com.qihoo360.mobilesafe.ui.index.AppEnterActivity");
                    intent.setComponent(comp);
                    break;
//                case "ZTE":
//
//                    break;
//                case "LENOVO":
//
//                    break;
//                case "VIVO":
//
//                    break;
//                case "SAMSUNG":
//
//                    break;
                case "MEIZU":
                    intent = new Intent("com.meizu.safe.security.SHOW_APPSEC");
                    intent.addCategory(Intent.CATEGORY_DEFAULT);
                    break;
                default:
                    intent = new Intent(Settings.ACTION_SETTINGS);
                    break;
            }

            intent.putExtra("packageName", BuildConfig.APPLICATION_ID);
            context.startActivity(intent);

        } catch (Exception e) {
            context.startActivity(new Intent(Settings.ACTION_SETTINGS));
        }

    }


    public interface OnPermissionListener {
        /**
         * 允许权限
         */
        void onAllow(Permission permission);

        /**
         * 拒绝权限，下次再问
         */
        void onRefuse(Permission permission);

        /**
         * 拒绝权限，且不要再问
         */
        void onRefuseAndNeverAskAgain(Permission permission);

        /**
         * 操作结束
         *
         * @param allowed  被授权的权限
         * @param refused  被拒绝的权限
         * @param neverAsk 被拒绝且选中不要再问的权限
         */
        void onFinish(Permission[] allowed, Permission[] refused, Permission[] neverAsk);
    }

    public abstract static class OnSimplePermissionListener implements OnPermissionListener {
        @Override
        public void onAllow(Permission permission) {

        }

        @Override
        public void onRefuse(Permission permission) {

        }

        @Override
        public void onRefuseAndNeverAskAgain(Permission permission) {

        }
    }
}
