package tech.starwin.base;

import android.content.Context;
import android.support.v7.widget.RecyclerView;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * Created by SiKang on 2018/9/16.
 */
public abstract class BaseRecyclerAdapter<T> extends RecyclerView.Adapter<RecyclerViewHolder> {
    private final List<T> mData;
    private final Context context;
    private OnItemClickListener<T> mClickListener;
    private OnItemLongClickListener<T> mLongClickListener;
    private List<Integer> childClickSet;

    public BaseRecyclerAdapter(Context ctx, List<T> list) {
        mData = (list != null) ? list : new ArrayList<T>();
        context = ctx;
    }

    @Override
    public RecyclerViewHolder onCreateViewHolder(ViewGroup parent, int viewType) {
        View view = null;
        if (bindItemLayout() instanceof Integer) {
            view = LayoutInflater.from(context).inflate((Integer) bindItemLayout(), parent, false);
        } else if (bindItemLayout() instanceof View) {
            view = (View) bindItemLayout();
        }
        if (view == null) {
            throw new RuntimeException("bindItemLayout()  invalid");
        }
        final RecyclerViewHolder holder = new RecyclerViewHolder(view);
        initViewHolder(holder);
        if (mClickListener != null) {
            holder.itemView.setOnClickListener(v ->
                    mClickListener.onItemClick(holder.itemView, holder.getLayoutPosition(), mData.get(holder.getLayoutPosition())));
        }
        if (mLongClickListener != null) {
            holder.itemView.setOnLongClickListener(v -> {
                mLongClickListener.onItemLongClick(holder.itemView, holder.getLayoutPosition(), mData.get(holder.getLayoutPosition()));
                return true;
            });
        }
        return holder;
    }

    @Override
    public final void onBindViewHolder(RecyclerViewHolder holder, int position) {
        bindData(holder, position, mData.get(position));
        if (mClickListener != null && childClickSet.size() > 0) {
            for (int id : childClickSet) {
                View child = holder.getView(id);
                if (child != null) {
                    child.setOnClickListener(v -> mClickListener.onItemClick(v, position, mData.get(position)));
                }
            }
        }
    }


    @Override
    public int getItemCount() {
        return mData.size();
    }

    /**
     * item布局
     */
    public abstract Object bindItemLayout();

    /**
     * item更新
     */
    public abstract void bindData(RecyclerViewHolder holder, int position, T item);

    public void initViewHolder(RecyclerViewHolder holder) {
    }

    public T getItem(int pos) {
        return mData.get(pos);
    }


    public void setOnItemClickListener(OnItemClickListener<T> listener, Integer... viewIds) {
        mClickListener = listener;
        childClickSet = new ArrayList<>();
        childClickSet.addAll(Arrays.asList(viewIds));

    }

    public void setOnItemLongClickListener(OnItemLongClickListener<T> listener) {
        mLongClickListener = listener;
    }


    public interface OnItemClickListener<T> {
        void onItemClick(View itemView, int pos, T t);
    }

    public interface OnItemLongClickListener<T> {
        void onItemLongClick(View itemView, int pos, T t);
    }

    public Context getContext() {
        return context;
    }
}

