package tech.starwin.impl;

import com.google.gson.Gson;

import io.reactivex.Observer;
import io.reactivex.disposables.Disposable;
import retrofit2.HttpException;
import tech.starwin.R;
import tech.starwin.LibConfig;
import tech.starwin.mvp.beans.ResponseErrorBody;
import tech.starwin.network.Error;
import tech.starwin.utils.LoginManager;

/**
 * Created by SiKang on 2018/9/14.
 * HTTP请求回调
 */
public abstract class HttpObserver<T> implements Observer<T> {
    private Disposable disposable;

    public abstract void onStart();

    public abstract void onSuccess(T data);

    public abstract void onError(int code, String msg);

    public abstract void onFinish();


    @Override
    public void onSubscribe(Disposable d) {
        this.disposable = d;
        onStart();
    }

    @Override
    public void onNext(T tResult) {
        onFinish();
        onSuccess(tResult);
    }

    @Override
    public void onError(Throwable throwable) {
        onFinish();
        String msg = throwable.getMessage();
        try {
            if (throwable != null) {
                if (isHttpError(msg)) {
                    onError(Error.HTTP_ERROR, LibConfig.getContext().getString(R.string.text_net_error));
                } else if (throwable instanceof HttpException) {
                    HttpException exception = (HttpException) throwable;
                    if (!isCanHandle(exception)) {
                        String body = exception.response().errorBody().string();
                        ResponseErrorBody responseErrorBody = new Gson().fromJson(body, ResponseErrorBody.class);
                        onError(Error.SERVER_ERROR, responseErrorBody.getMessage());
                    }
                } else if (msg.contains("java.lang.IllegalStateException")) {
                    onError(Error.APP_ERROR, LibConfig.getContext().getString(R.string.text_app_error));
                } else if (msg.contains("body is null")) {
                    onSuccess(null);
                } else {
                    onError(Error.UNKNOW, msg);
                }

            }
        } catch (Throwable e) {
            onError(Error.APP_ERROR, msg);
        }
        if (disposable != null)
            disposable.dispose();
    }

    @Override
    public void onComplete() {
        if (disposable != null)
            disposable.dispose();
    }

    /**
     * 可预处理的错误
     */
    private boolean isCanHandle(HttpException exception) {
        switch (exception.code()) {
            case 401://Token失效
                LoginManager.get().tokenInvalid();
                return true;
            case 403:
                // TODO 短信发送失败
                return true;
            case 409:
                // TODO 版本更新
                return true;
            case 412:
                // TODO 参数缺失
                return true;
            case 404:
            case 500:
                onError(Error.SERVER_ERROR, "server exception");
                return true;
        }
        return false;
    }

    /**
     * 链路错误
     */
    private boolean isHttpError(String msg) {
        String[] errorArray = new String[]{
                "timeout",
                "java.net.ConnectException",
                "java.net.SocketTimeoutException",
                "failed",
                "Failed to connect to",
                "stream was reset",
                "Unable to resolve host",
                "SSL handshake time out",
                "time out"
        };
        for (String error : errorArray) {
            if (msg.contains(error))
                return true;
        }
        return false;
    }

}
