package com.common.base;

import android.app.Activity;
import android.content.Context;
import android.support.annotation.CheckResult;
import android.support.annotation.IdRes;
import android.support.annotation.NonNull;
import android.support.annotation.StringRes;
import android.view.View;
import android.view.ViewGroup;

import com.trello.rxlifecycle2.LifecycleTransformer;
import com.trello.rxlifecycle2.RxLifecycle;
import com.trello.rxlifecycle2.android.ActivityEvent;
import com.trello.rxlifecycle2.android.RxLifecycleAndroid;


import io.reactivex.Observable;
import io.reactivex.subjects.BehaviorSubject;
import tech.starwin.LibConfig;
import tech.starwin.base.BasePresenter;
import tech.starwin.mvp.IView;
import tech.starwin.utils.PresenterHoler;
import tech.starwin.utils.context_utils.FragmentLauncher;

/**
 * Created by SiKang on 2019/3/12.
 */
public abstract class BaseTrustee implements IView {
    private ViewGroup trustorView;
    private Activity trustorActivity;
    private Object unBinder;
    /**
     * 创建和缓存Presenter实例
     */
    private PresenterHoler presenterHelper;
    private FragmentLauncher fragmentLauncher;

    private final BehaviorSubject<ActivityEvent> lifecycleSubject = BehaviorSubject.create();

    public BaseTrustee(Activity activity, ViewGroup view) {
        this.trustorActivity = activity;
        this.trustorView = view;
    }

    protected View findViewById(@IdRes int viewId) {
        return trustorView.findViewById(viewId);
    }

    public abstract void initView();

    public void setPresenterHelper(PresenterHoler presenterHelper) {
        this.presenterHelper = presenterHelper;
    }

    /**
     * 获取Presenter
     */
    public <T extends BasePresenter> T getPresenter(Class<T> clz) {
        if (presenterHelper == null) {
            presenterHelper = new PresenterHoler(this);
        }
        return presenterHelper.getPresenter(clz);
    }

    /**
     * 生命周期托管
     */
    public void onCreate() {
        lifecycleSubject.onNext(ActivityEvent.CREATE);
        unBinder = LibConfig.bindView(this, trustorView);
        initView();
    }

    protected void onStart() {
        lifecycleSubject.onNext(ActivityEvent.START);
    }

    protected void onRestart() {
    }

    protected void onResume() {
        lifecycleSubject.onNext(ActivityEvent.RESUME);
    }

    protected void onPause() {
        lifecycleSubject.onNext(ActivityEvent.PAUSE);
    }

    protected void onStop() {
        lifecycleSubject.onNext(ActivityEvent.STOP);
    }

    protected void onDestroy() {
        lifecycleSubject.onNext(ActivityEvent.DESTROY);
        if (unBinder != null) {
            LibConfig.unBindView(unBinder);
        }
        if (presenterHelper != null) {
            presenterHelper.onDestory();
        }
    }

    public void onBackPressed(){}


    @Override
    public void onHttpStart(String action, boolean isShowLoading) {

    }

    @Override
    public void onHttpSuccess(String action, Object result) {

    }

    @Override
    public void onHttpError(String action, String msg) {

    }

    @Override
    public void onHttpFinish(String action) {

    }

    @Override
    @NonNull
    @CheckResult
    public final Observable<ActivityEvent> lifecycle() {
        return lifecycleSubject.hide();
    }

    @Override
    @NonNull
    @CheckResult
    public final <T> LifecycleTransformer<T> bindUntilEvent(@NonNull ActivityEvent event) {
        return RxLifecycle.bindUntilEvent(lifecycleSubject, event);
    }

    @Override
    @NonNull
    @CheckResult
    public final <T> LifecycleTransformer<T> bindToLifecycle() {
        return RxLifecycleAndroid.bindActivity(lifecycleSubject);
    }


    public Activity getActivity() {
        return trustorActivity;
    }

    public ViewGroup getTrustorView() {
        return trustorView;
    }

    public String getString(@StringRes int id) {
        return getActivity().getString(id);
    }

    public FragmentLauncher getFragmentLauncher() {
        return fragmentLauncher;
    }

    public void setFragmentLauncher(FragmentLauncher fragmentLauncher) {
        this.fragmentLauncher = fragmentLauncher;
    }
}
