package tech.starwin.base;

import android.content.Context;
import android.support.v7.widget.RecyclerView;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by SiKang on 2018/9/16.
 */
public abstract class BaseRecyclerAdapter<T> extends RecyclerView.Adapter<RecyclerViewHolder> {
    private final List<T> mData;
    private final Context context;
    private OnItemClickListener<T> mClickListener;
    private OnItemLongClickListener<T> mLongClickListener;

    public BaseRecyclerAdapter(Context ctx, List<T> list) {
        mData = (list != null) ? list : new ArrayList<T>();
        context = ctx;
    }

    @Override
    public RecyclerViewHolder onCreateViewHolder(ViewGroup parent, int viewType) {
        View view = null;
        if (setItemLayout() instanceof Integer) {
            view = LayoutInflater.from(context).inflate((Integer) setItemLayout(), parent, false);
        } else if (setItemLayout() instanceof View) {
            view = (View) setItemLayout();
        }
        if (view == null) {
            throw new RuntimeException("setItemLayout()  invalid");
        }
        final RecyclerViewHolder holder = new RecyclerViewHolder(view);

        if (mClickListener != null) {
            holder.itemView.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    mClickListener.onItemClick(holder.itemView, holder.getLayoutPosition(), mData.get(holder.getLayoutPosition()));
                }
            });
        }
        if (mLongClickListener != null) {
            holder.itemView.setOnLongClickListener(new View.OnLongClickListener() {
                @Override
                public boolean onLongClick(View v) {
                    mLongClickListener.onItemLongClick(holder.itemView, holder.getLayoutPosition(), mData.get(holder.getLayoutPosition()));
                    return true;
                }
            });
        }
        return holder;
    }

    @Override
    public void onBindViewHolder(RecyclerViewHolder holder, int position) {
        bindData(holder, position, mData.get(position));
    }


    @Override
    public int getItemCount() {
        return mData.size();
    }

    /**
     * item布局
     */
    public abstract Object setItemLayout();

    /**
     * item更新
     */
    public abstract void bindData(RecyclerViewHolder holder, int position, T item);

    public T getItem(int pos) {
        return mData.get(pos);
    }


    public void setOnItemClickListener(OnItemClickListener listener) {
        mClickListener = listener;
    }

    public void setOnItemLongClickListener(OnItemLongClickListener listener) {
        mLongClickListener = listener;
    }


    public interface OnItemClickListener<T> {
        void onItemClick(View itemView, int pos, T t);
    }

    public interface OnItemLongClickListener<T> {
        void onItemLongClick(View itemView, int pos, T t);
    }

    public Context getContext() {
        return context;
    }
}

