package tech.starwin.utils;

import android.app.Activity;
import android.content.Context;
import android.content.SharedPreferences;
import android.text.TextUtils;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;

import org.json.JSONException;
import org.json.JSONStringer;

import java.util.HashMap;
import java.util.Map;

import tech.starwin.mvp.beans.GatewayInfoBean;
import tech.starwin.mvp.beans.OcrResultBean;
import tech.starwin.mvp.beans.TokenInfoBean;
import tech.starwin.mvp.beans.UserBean;
import tech.starwin.utils.format_utils.DataFormat;

/**
 * Created by SiKang on 2018/9/18.
 * SharePreferences 管理
 */
public class PreferencesManager {
    private static final String PREFERENCES_FILE_NAME = "base_info";
    //单例
    private static PreferencesManager preferencesManager;
    // 获取SharedPreferences对象
    private SharedPreferences mSharedPreferences;
    // 获取Editor对象
    private SharedPreferences.Editor mEditor;
    private Context context;

    private PreferencesManager() {

    }

    public void init(Context context) {
        this.context = context;
        mSharedPreferences = context.getSharedPreferences(PREFERENCES_FILE_NAME, Activity.MODE_PRIVATE);
        mEditor = mSharedPreferences.edit();
    }

    public static PreferencesManager get() {
        if (preferencesManager == null) {
            synchronized (PreferencesManager.class) {
                if (preferencesManager == null) {
                    preferencesManager = new PreferencesManager();
                }
            }
        }
        return preferencesManager;
    }

    /**
     * 存取登录信息
     */
    public void saveTokenInfo(TokenInfoBean tokenInfoBean) {
        saveData("token_info", DataFormat.fromBean(tokenInfoBean));
    }

    public TokenInfoBean getTokenInfo() {
        String objStr = getString("token_info", "");
        return TextUtils.isEmpty(objStr) ? null : (TokenInfoBean) DataFormat.toBean(objStr);
    }


    /**
     * 存取用户信息
     */
    public void saveUserInfo(UserBean userInfo) {
        saveData("user_info", DataFormat.fromBean(userInfo));
    }

    public UserBean getUserInfo() {
        String objStr = getString("user_info", "");
        return TextUtils.isEmpty(objStr) ? null : (UserBean) DataFormat.toBean(objStr);
    }


    /**
     * 存取KTP信息
     */
    public void saveKTPInfo(OcrResultBean.KTP ktpInfo) {
        saveData("ktp_info", DataFormat.fromBean(ktpInfo));
    }

    public OcrResultBean.KTP getKTPInfo() {
        String objStr = getString("ktp_info", "");
        return TextUtils.isEmpty(objStr) ? null : (OcrResultBean.KTP) DataFormat.toBean(objStr);
    }


    /**
     * 存取Gateway信息
     */
    public void saveGatewayInfo(GatewayInfoBean gatewayInfo) {
        saveData("gateway_info", DataFormat.fromBean(gatewayInfo));
    }

    public GatewayInfoBean getGatewayInfo() {
        String objStr = getString("gateway_info", "");
        return TextUtils.isEmpty(objStr) ? null : (GatewayInfoBean) DataFormat.toBean(objStr);
    }


    public void saveDefaultSmsPackage(String pkgName) {
        saveData("default_sms_pkg", pkgName);
    }

    public String getDefaultSmsPackage() {
        return getString("default_sms_pkg", "");
    }

    /**
     * 存取依图活体识别截图
     */
    public void saveVerificationData(byte[] data) {
        if (data == null) {
            saveData("verification_data", "");
        } else {
            saveData("verification_data", DataFormat.base64(data));
        }
    }

    public String getVerificationData() {
        return getString("verification_data", null);
    }



    /**
     * 更新政策 是否同意 状态
     */
    public void setPolicyStatus(boolean isAgreed) {
        saveData("has_agreed_policy", isAgreed);
    }

    /**
     * 保存上传用户数据时使用的sessionId
     */
    public void saveSessionId(String sessionId) {
        saveData("session_id", sessionId);
    }

    /**
     * 取出上传用户数据时使用的sessionId
     */
    public String getSessionId() {
        return getString("session_id", "");
    }

    /**
     * 是否已同意政策
     */
    public boolean hasAgreedPolicy() {
        return getBoolean("has_agreed_policy", false);
    }


    /**
     * 清除登录信息
     */
    public void clearLoginInfo() {
        mEditor.remove("user_info");
        mEditor.remove("token_info");
        mEditor.commit();
    }

    /**
     * 保存单条数据
     */
    public void saveData(String key, String data) {
        mEditor.putString(key, data);
        mEditor.commit();
    }

    /**
     * 保存单条数据(int)
     */
    public void saveData(String key, int data) {
        mEditor.putInt(key, data);
        mEditor.commit();
    }


    /**
     * 保存单条数据(boolean)
     */
    public void saveData(String key, boolean flag) {
        mEditor.putBoolean(key, flag);
        mEditor.commit();
    }


    /**
     * 删除一条数据
     */
    public void remove(String key) {
        mEditor.remove(key);
        mEditor.commit();
    }

    /**
     * 保存多条数据
     */
    public void saveData(Map<String, String> map) {
        for (String key : map.keySet()) {
            mEditor.putString(key, map.get(key));
        }
        mEditor.commit();
    }

    /**
     * 根据key取指定字段
     */
    public String getString(String key, String defaultValue) {
        return mSharedPreferences.getString(key, defaultValue);
    }

    /**
     * 根据key取指定字段(int)
     */
    public int getInt(String key, int defaultValue) {
        return mSharedPreferences.getInt(key, defaultValue);
    }

    /**
     * 根据key取指定字段(boolean)
     */
    public boolean getBoolean(String key, boolean defaultValue) {
        return mSharedPreferences.getBoolean(key, defaultValue);
    }

    /**
     * 清除数据
     */
    public void clearAll() {
        mEditor = mSharedPreferences.edit();
        mEditor.clear();
        mEditor.commit();
    }

    /**
     * 查询某个key是否已经存在
     *
     * @param key
     * @return
     */
    public boolean contains(String key) {
        return mSharedPreferences.contains(key);
    }

}
